# Supplementary Areas Plugin

A BriefConnect plugin that adds a custom tab for managing supplementary areas on Cabinet and Coordination Comment briefing records.

## Overview

This plugin provides a rich interface for managing multiple supplementary areas associated with briefing records. Each supplementary area tracks executive directors, allocators, contributors, comments, and endorsement status. The plugin synchronizes data between plugin storage and SharePoint fields, gracefully handling missing or misconfigured fields.

## Features

### Core Functionality
- **Multiple Areas**: Add up to 20 supplementary areas per record
- **People Management**: Search and assign up to 50 users per field (Executive Directors, Allocators, Contributors)
- **Comments Thread**: Conversation-style comment history with timestamps
- **Endorsement Tracking**: Checkbox to mark area endorsement as complete
- **Due Date**: Global due date field for all supplementary areas
- **Auto-sync**: Automatic synchronization with SharePoint fields

### User Experience
- **Fluent UI Design**: Modern interface using Microsoft Fluent UI components
- **Accordion Layout**: Expandable/collapsible area panels
- **Real-time Search**: Debounced people search with keyboard navigation
- **Change Tracking**: Dirty state detection with save/cancel operations
- **Error Handling**: Graceful handling of missing fields with user-friendly notifications
- **Validation**: User limits, field constraints, and stale value detection

### Technical Features
- **Shadow DOM**: Encapsulated styles and markup
- **Web Component**: Custom element (`<supplementary-areas-tab>`)
- **Optimistic Updates**: Minimal re-renders for better performance
- **Accessibility**: ARIA labels, keyboard navigation, screen reader support
- **Version Control**: Optimistic concurrency with version tracking

## Supported Record Types

- Cabinet or Committee Briefing (`CabBRI`)
- Coordination Comment Briefing (`CabCRD`)

## Installation

### Prerequisites
- BriefConnect Plugin SDK
- Azure Function App for backend API
- SharePoint content type with appropriate fields

### Deployment
1. Deploy the plugin to your Azure Function App
2. Ensure the plugin is registered in the BriefConnect system
3. Configure SharePoint content types with required fields (see Configuration section)

## Configuration

### SharePoint Fields

The plugin expects the following field naming conventions for synchronization:

#### Global Fields
- `DueDate` - Date field for global due date

#### Per-Area Fields (indexed 1-20)
- `SupplementaryAreaName{N}` - Text field for area label
- `ExecutiveDirectors{N}` - Person or Group field (multi-select)
- `Allocators{N}` - Person or Group field (multi-select)
- `Contributors{N}` - Person or Group field (multi-select)
- `Comments{N}` - Note/Multi-line text field
- `SupplementaryAreaEndorsement{N}` - Boolean/Checkbox field

Where `{N}` is the area index (1-20).

### Dropdown Configuration

Area names are populated from a dropdown configuration. Configure this in your backend plugin settings.

### Missing Fields

The plugin gracefully handles missing fields:
- **Plugin Storage**: Data is always saved in plugin storage
- **SharePoint Sync**: Only configured fields are synchronized to SharePoint
- **User Notification**: Missing fields are displayed with warning messages
- **Data Preservation**: No data loss occurs if fields are missing

## Usage

### Adding a Supplementary Area
1. Open a supported briefing record
2. Navigate to the "Supplementary Areas" tab
3. Click "New supplementary area"
4. Fill in the area details
5. Click "Save"

### Managing People
1. Click in a people picker field (Executive Directors, Allocators, Contributors)
2. Type at least 2 characters to search
3. Select people from the dropdown
4. Remove people by clicking the × button on their badge

### Adding Comments
1. Expand a supplementary area
2. Scroll to the Comments section
3. Type your comment in the text area
4. Click "Save" to persist the comment

### Endorsement
1. Check the "Mark Supplementary Area Endorsement as complete" checkbox
2. Click "Save"

### Deleting an Area
1. Expand the area you want to delete
2. Click the "Delete" button (trash icon)
3. Confirm the deletion
4. Click "Save"

## Architecture

### Frontend (Web Component)

**File**: `ui/main.js`

The plugin uses a Web Component (`SupplementaryAreasTab`) with Shadow DOM for encapsulation.

#### Component Lifecycle
1. **Construction**: Initialize state, inject styles
2. **Connection**: Parse props, load data
3. **Rendering**: Build HTML from state
4. **Event Handling**: Attach listeners for user interactions
5. **Cleanup**: Remove listeners and timers

#### State Management
- `_model`: Main data model (areas, dueDate, version)
- `_record`: Record summary and permissions
- `_fields`: Field metadata from backend
- `_dirtyValues`: Change tracking flag
- `_baselineValuesHash`: Hash for dirty detection

#### Key Methods
- `_loadInitialData()`: Fetch data from backend
- `saveAll()`: Persist complete state to backend
- `addArea()`: Create new supplementary area
- `deleteArea()`: Remove existing area
- `updateArea()`: Update area properties
- `_searchPeople()`: Search for users

### Backend API

The plugin backend should implement the following endpoints:

#### GET (Load Data)
**Query Parameters:**
- `recordId`: The record ID to load

**Response:**
```json
{
  "model": {
    "version": "string",
    "dueDate": "2025-10-16T00:00:00Z",
    "areas": [
      {
        "id": "uuid",
        "index": 1,
        "order": 1,
        "label": "Area Name",
        "executiveDirectors": [...],
        "allocators": [...],
        "contributors": [...],
        "comments": [
          {
            "id": "uuid",
            "text": "Comment text",
            "createdUtc": "2025-10-21T10:00:00Z",
            "author": {
              "displayName": "John Doe"
            }
          }
        ],
        "endorsementComplete": false
      }
    ]
  },
  "record": {
    "recordId": "123",
    "recordType": "Cabinet or Committee Briefing (CabBRI)",
    "canUserEditMetadata": true
  },
  "fields": [...],
  "dropdownOptions": [
    { "id": "value1", "label": "Display Name 1" }
  ],
  "dueDateFieldInternalName": "DueDate",
  "missingFields": {
    "global": ["FieldName1", "FieldName2"]
  }
}
```

#### POST updateRecord (Save Data)
**Query Parameters:**
- `recordId`: The record ID to update
- `recordType`: The record type

**Request Body:**
```json
{
  "version": "string",
  "dueDate": "2025-10-16",
  "recordType": "Cabinet or Committee Briefing (CabBRI)",
  "areas": [
    {
      "id": "uuid",
      "label": "Area Name",
      "order": 1,
      "index": 1,
      "executiveDirectors": [...],
      "allocators": [...],
      "contributors": [...],
      "comments": [...],
      "newComment": "Optional new comment text",
      "endorsementComplete": false
    }
  ]
}
```

**Response:**
```json
{
  "model": { ... },
  "modelVersion": "updated-version-string"
}
```

#### GET (People Search)
**Query Parameters:**
- `recordId`: Current record ID
- `action`: `peopleSearch`
- `q`: Search query

**Response:**
```json
{
  "results": [
    {
      "GraphUserId": "uuid",
      "DisplayName": "John Doe",
      "Mail": "john.doe@example.com",
      "UserPrincipalName": "john.doe@example.com",
      "JobTitle": "Manager"
    }
  ]
}
```

## Constants and Limits

```javascript
MAX_AREAS: 20                    // Maximum supplementary areas per record
MAX_USERS_PER_FIELD: 50         // Maximum users per people picker field
FLASH_MESSAGE_TTL: 3500         // Notification duration (ms)
SEARCH_DEBOUNCE_DELAY: 250      // Search input debounce (ms)
```

## Development

### Prerequisites
- Node.js (for any build tools)
- Modern browser with Web Components support
- BriefConnect development environment

### Local Development
1. Edit `ui/main.js` for frontend changes
2. Test in a development BriefConnect instance
3. Use browser DevTools to debug Shadow DOM

### Code Structure

```
BriefConnect.Plugin.DE.SupplementaryArea/
├── ui/
│   └── main.js          # Frontend Web Component
├── README.md            # This file
└── [backend files]      # C# Azure Functions
```

### Key Classes and Interfaces

**SupplementaryAreasTab** - Main web component class
- Properties: `_props`, `_model`, `_record`, `_fields`
- Methods: `canEdit()`, `addArea()`, `deleteArea()`, `updateArea()`, `saveAll()`

**Area Object Structure**:
```javascript
{
  id: string,           // Unique identifier
  index: number,        // Display order
  order: number,        // Alternative order field
  label: string,        // Area name
  executiveDirectors: Array<Person>,
  allocators: Array<Person>,
  contributors: Array<Person>,
  comments: Array<Comment>,
  endorsementComplete: boolean
}
```

**Person Object Structure**:
```javascript
{
  GraphUserId: string,
  DisplayName: string,
  Mail: string,
  UserPrincipalName: string,
  JobTitle: string
}
```

## Browser Support

- Modern browsers with Web Components support (Chrome, Edge, Firefox, Safari)
- Shadow DOM support required
- ES6+ JavaScript support required

## Security

- Permissions checked via `canUserEditMetadata` flag
- Backend validation for all mutations
- CSRF protection through BriefConnect framework
- User context from authenticated session

## Performance Optimizations

- Debounced search input (250ms)
- Incremental rendering for large people lists
- Minimal re-renders (updates without destroying DOM)
- Hash-based dirty detection
- Cleanup of event listeners and timers

## Accessibility

- ARIA labels and roles
- Keyboard navigation support
- Screen reader announcements
- Focus management
- High contrast mode support

## Troubleshooting

### Missing Fields Warning
**Issue**: Warning about missing fields in content type  
**Solution**: Contact SharePoint administrator to add the required fields to the content type

### People Search Not Working
**Issue**: No results when searching for people  
**Solution**: Ensure backend people search API is configured and Graph API permissions are granted

### Save Button Disabled
**Issue**: Cannot save changes  
**Solution**: Make a change to enable the save button, or check browser console for errors

### Areas Not Loading
**Issue**: "No data" message displayed  
**Solution**: Check backend API connectivity and ensure record ID is valid